<?php

/**
 * The file that defines the core plugin class.
 *
 * Class definition that includes attributes and function used
 * across the public-facing side of the site and admin area.
 *
 * @link        http://ecrosio.com
 * @since       1.0.0
 *
 * @package     Ecrosio_Rescal
 * @subpackage  Ecrosio_Rescal/includes
 */

/**
 * The core plugin class.
 *
 * This is used to define the internationalization, admin and public-facing
 * site hooks.
 * Also maintains the unique identifier and version of the plugin.
 *
 * @since       1.0.0
 * @package     Ecrosio_Rescal
 * @subpackage  Ecrosio_Rescal/includes
 * @author      egortar <y.tarkovskyi@ecrosio.com>
 */
class Ecrosio_Rescal {

    /**
     * The loader that is responsible for maintaining and registering
     * all plugin's hooks and actions.
     *
     * @since   1.0.0
     * @access  protected
     * @var     Ecrosio_Rescal_Loader $loader Maintains and registers all hooks and actions for the plugin.
     */
    protected $loader;

    /**
     * The unique identifier of the plugin.
     *
     * @since   1.0.0
     * @access  protected
     * @var     string $plugin_name This string contains the unique identifier of this plugin.
     */
    protected $plugin_name;

    /**
     * Current version of the plugin.
     *
     * @since   1.0.0
     * @access  protected
     * @var     string $version The current version of the plugin.
     */
    protected $version;

    /**
     * Plugin's core
     *
     * @since   1.0.0
     * @access  protected
     * @var     Ecrosio_Rescal_Core $core Plugin's core
     */
    protected $core;

    /**
     * Define the core functionality of the plugin
     *
     * Set the plugin name and version, load the dependencies, define the locale and set hooks for
     * admin and public areas of the site.
     *
     * @since 1.0.0
     */
    public function __construct() {
        if ( defined( 'PLUGIN_VERSION' ) ) {
            $this->version = PLUGIN_VERSION;
        } else {
            $this->version = '1.0.0';
        }
        $this->plugin_name = 'Ecrosio-Rescal';

        $this->load_dependencies();
        $this->set_locale();
        $this->define_admin_hooks();
    }

    /**
     * Load the required dependencies for this plugin.
     *
     * Create an instance of the loader which will be used to register hooks
     * and actions with WordPress.
     *
     * @since   1.0.0
     * @access  private
     */
    private function load_dependencies() {

        /**
         * The class responsible for working with the actions and filters of the core plugin.
         */
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-ecrosio-rescal-loader.php';

        /**
         * The class responsible for defining internationalization functionality
         * of the plugin.
         */
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-ecrosio-rescal-i18n.php';


        /**
         * The class responsible for plugin's core functionality that is not connected with DIVI PageBuilder
         */
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'core/class-ecrosio-rescal-core.php';


        $this->loader = new Ecrosio_Rescal_Loader();
        $this->core   = new Ecrosio_Rescal_Core();
    }

    /**
     * Define the locale for this plugin for internationalization.
     *
     * Uses the Ecrosio_Rescal_i18n class in order to set the domain and to register the hook
     * with WordPress.
     *
     * @since    1.0.0
     * @access   private
     */
    private function set_locale() {
        $plugin_i18n = new Ecrosio_Rescal_i18n();
        $this->loader->add_action( 'plugins_loaded', $plugin_i18n, 'load_plugin_textdomain' );
    }

    /**
     * Register all of the hooks related to the admin area functionality
     * of the plugin.
     *
     * @since    1.0.0
     * @access   private
     */
    private function define_admin_hooks() {
    }

    /**
     * Run the loader to execute all of the hooks with WordPress.
     *
     * @since    1.0.0
     */
    public function run() {
        $this->loader->run();
    }

    /**
     * The name of the plugin used to uniquely identify it within the context of
     * WordPress and to define internationalization functionality.
     *
     * @since     1.0.0
     * @return    string    The name of the plugin.
     */
    public function get_plugin_name() {
        return $this->plugin_name;
    }

    /**
     * The reference to the class that orchestrates the hooks with the plugin.
     *
     * @since     1.0.0
     * @return    Ecrosio_Rescal_Loader    Orchestrates the hooks of the plugin.
     */
    public function get_loader() {
        return $this->loader;
    }

    /**
     * Retrieve the version number of the plugin.
     *
     * @since     1.0.0
     * @return    string    The version number of the plugin.
     */
    public function get_version() {
        return $this->version;
    }

    /**
     * Get instance of plugin's core
     *
     * @since   1.0.0
     * @return  Ecrosio_Rescal_Core  Plugin's core
     */
    public function get_core() {
        return $this->core;
    }
}