<?php

/**
 * The file that defines dataHandler class which is used for file and DB I\O operations.
 *
 * Class definition that includes attributes and function used for data handling functionality
 *
 * @link        http://ecrosio.com
 * @since       1.0.0
 *
 * @package     Ecrosio_Rescal
 * @subpackage  Ecrosio_Rescal/core
 */

/**
 * Data handler class.
 *
 * This is used to store and read data that is necessary for calendar's functionality
 *
 * @since       1.0.0
 * @package     Ecrosio_Rescal
 * @subpackage  Ecrosio_Rescal/core
 * @author      egortar <y.tarkovskyi@ecrosio.com>
 */
class Ecrosio_Rescal_DataHandler {

    /**
     * List of added calendars
     * @access  protected
     * @var     array   List of added calendars
     */
    protected $calendarList;

    /**
     * Ecrosio_Rescal_DataHandler constructor.
     */
    public function __construct() {
    }

    /**
     * Load calendars list from JSON file
     *
     * @since   1.0.0
     *
     * @param   boolean $active Select only active calendars. Default : false
     *
     * @return  array   $calendars  List of saved calendars
     */
    public static function load_calendars( $active = false ) {

        $json_string = file_get_contents( PLUGIN_DATA_DIR . 'calendars.json' );
        $calendars   = json_decode( $json_string, true );

        if ( $active ) {
            $calendars_active = array();

            foreach ( $calendars as $calendar ) {
                if ( $calendar['status'] == "active" ) {
                    array_push( $calendars_active, $calendar );
                }
            }

            return $calendars_active;
        }

        return $calendars;
    }

    /**
     * Save calendars list into file
     *
     * @since   1.0.0
     *
     * @param   array $calendars List of calendars to save
     *
     * @return  boolean $status     Operation status
     */
    public static function save_calendars( $calendars ) {
        $json = json_encode( $calendars );

        if ( file_put_contents( PLUGIN_DATA_DIR . 'calendars.json', $json ) ) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * Select calendar by it's name
     *
     * @since   1.0.0
     *
     * @param   string $name Calendar name
     *
     * @return  array   $calendar   Parameters of selected calendar
     */
    public static function select_calendar_name( $name ) {

        $calendars = self::load_calendars();

        foreach ( $calendars as $calendar ) {
            if ( $calendar['name'] == $name ) {
                return $calendar;
            }
        }
    }

    /**
     * Select calendar by it's id
     *
     * @since   1.0.0
     *
     * @param   int $id Calendar id
     *
     * @return  array   $calendar   Parameters of selected calendar
     */
    public static function select_calendar_id( $id ) {

        $calendars = self::load_calendars();

        foreach ( array_keys( $calendars ) as $key ) {
            if ( $key == $id ) {
                return $calendars[ $key ];
            }
        }
    }

    /**
     * Get calendar's name by it's id
     *
     * @since   1.0.0
     *
     * @param   int $id Calendar id
     *
     * @return  string $name Name of selected calendar.
     */
    public static function get_calendar_name( $id ) {

        $calendars = self::load_calendars();

        foreach ( array_keys( $calendars ) as $key ) {
            if ( $key == $id ) {
                return $calendars[ $key ]['name'];
            }
        }
    }

    /**
     * Add new calendar into list
     *
     * @since   1.0.0
     *
     * @param   string $name Name of the calendar
     *
     * @return boolean  $isSaved    Operation status
     */
    public static function add_calendar( $name ) {

        $calendars = self::load_calendars();
        $calendar  = array( "name" => $name, "status" => "active" );

        array_push( $calendars, $calendar );

        if ( self::save_calendars( $calendars ) ) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * Remove calendar from list by it's id
     *
     * @since   1.0.0
     *
     * @param   int $id Id of calendar to remove
     *
     * @return  boolean Operation status
     */
    public static function remove_calendar_id( $id ) {

        $calendars = self::load_calendars();

        foreach ( array_keys( $calendars ) as $key ) {
            if ( $key == $id ) {
                unset( $calendars[ $key ] );
            }
        }

        if ( self::save_calendars( $calendars ) ) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * Remove calendar from list by it's name
     *
     * @since   1.0.0
     *
     * @param   string $name Name of calendar to remove
     *
     * @return  boolean Operation status
     */
    public static function remove_calendar_name( $name ) {

        $calendars = self::load_calendars();
        foreach ( $calendars as $calendar ) {
            if ( $calendar['name'] == $name ) {
                unset( $calendars[ array_search( $calendar, $calendars ) ] );
            }
        }
        if ( self::save_calendars( $calendars ) ) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * Edit calendar's parameters by it's id
     *
     * @since   1.0.0
     *
     * @param   int $id Id of calendar to edit
     * @param   array $params New calendar's params
     *
     * @return  boolean Operation status.
     */
    public static function edit_calendar_id( $id, $params ) {
        $calendars = self::load_calendars();

        foreach ( array_keys( $calendars ) as $key ) {
            if ( $key == $id ) {
                //SET $params for $calendar[$key]
            }
        }
        if ( self::save_calendars( $calendars ) ) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * Edit calendar's parameters by it's name
     *
     * @since   1.0.0
     *
     * @param   string $name Name if calendar to edit
     * @param   array $params New calendar's params
     *
     * @return  boolean Operation status
     */
    public static function edit_calendar_name( $name, $params ) {

        $calendars = self::load_calendars();
        foreach ( $calendars as $calendar ) {
            if ( $calendar['name'] == $name ) {
                //SET $params for $calendar
            }
        }
        if ( self::save_calendars( $calendars ) ) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * Get list of calendars names
     *
     * @since 1.0.0
     *
     * @param boolean $active Select only active calendars. Default: false
     *
     * @return array List of calendars names
     */
    public static function load_calendars_names( $active = false ) {

        $json_string = file_get_contents( PLUGIN_DATA_DIR . 'calendars.json' );
        $calendars   = json_decode( $json_string, true );
        $names       = array();

        if ( $active ) {
            foreach ( $calendars as $calendar ) {
                if ( $calendar['status'] == "active" ) {
                    array_push( $names, $calendar['name'] );
                }
            }

            return $names;
        }

        foreach ( $calendars as $calendar ) {
            array_push( $names, $calendar['name'] );
        }

        return $names;
    }

    /**
     * Get list of available timeslots for selected calendar
     *
     * @since 1.0.0
     *
     * @param int $id Selected calendar id. Default: null
     *
     * @return array List of available timeslots for selected calendar (or all timeslots if $id = null).
     */
    public static function load_timeslots( $id = null ) {

        $json_string = file_get_contents( PLUGIN_DATA_DIR . 'timeslots.json' );
        $timeslots   = json_decode( $json_string, true );

        if ( isset( $id ) ) {
            $selectedTimeslots = array();
            foreach ( $timeslots as $key => $timeslot ) {
                if ( $timeslot['calendarId'] == $id ) {
                    array_push( $selectedTimeslots, array( $key => $timeslot ) );
                }
            }

            return $selectedTimeslots;
        }

        return $timeslots;
    }

    /**
     * Delete timeslot with given id
     *
     * @since 1.0.0
     *
     * @param int $id Selected timeslot's id
     *
     * @return mixed Array of timeslots after removal.
     */
    public static function delete_timeslot( $id ) {

        $timeslots     = self::load_timeslots();
        $timeslots_new = array();

        foreach ( $timeslots as $key => $timeslot ) {
            if ( $key == $id ) {
                unset( $timeslot );
            } else {
                array_push( $timeslots_new, $timeslot );
            }
        }

        return $timeslots_new;
    }

    /**
     * Save changed timeslots list
     *
     * @since 1.0.0
     *
     * @param array $data New timeslots list
     *
     * @return boolean Operation status
     */
    public static function save_timeslots( $data ) {
        if ( file_put_contents( PLUGIN_DATA_DIR . 'timeslots.json', json_encode( $data ) ) ) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * Load saved calendar's DIVI module settings
     *
     * @since 1.0.0
     *
     * @param int $id Calendar's id
     *
     * @return array Array of selected calendar's settings.
     */
    public static function load_calendars_settings( $id ) {

        $json_string     = file_get_contents( PLUGIN_DATA_DIR . 'calendar_styles.json' );
        $calendar_styles = json_decode( $json_string, true );

        foreach ( $calendar_styles as $calendar_style ) {
            if ( $calendar_style['calendar_id'] == $id ) {
                return $calendar_style;
            }
        }

        return array();
    }

    /**
     * Save calendar's DIVI module settings.
     *
     * @since 1.0.0
     *
     * @param int $id Calendar's id.
     * @param array $settings List of settings
     *
     * @return boolean Operation status.
     */
    public static function save_calendars_settings( $id, $settings ) {

        $json_string     = file_get_contents( PLUGIN_DATA_DIR . 'calendar_styles.json' );
        $calendar_styles = json_decode( $json_string, true );
        $new             = true;

        foreach ( $calendar_styles as $key => $calendar_style ) {
            if ( $calendar_style['calendar_id'] == $id ) {
                $old                     = false;
                $calendar_styles[ $key ] = $settings;
            }
        }
        if ( $new ) {
            array_push( $calendar_styles, $settings );
        }

        if ( file_put_contents( PLUGIN_DATA_DIR . 'calendar_styles.json', json_encode( $calendar_styles ) ) ) {
            return true;
        } else {
            return false;
        }
    }
}